using System;
using System.Collections.Generic;

class TreapNode
{
    public int key, priority;
    public int count;      // multiplicity in multiset
    public long subSum;    // sum in subtree
    public int subCount;   // number of elements in subtree
    public TreapNode left, right;

    public TreapNode(int key)
    {
        this.key = key;
        priority = Random.Shared.Next();
        count = 1;
        Recalc();
    }

    public void Recalc()
    {
        subCount = count;
        subSum = (long)count * key;

        if (left != null)
        {
            subCount += left.subCount;
            subSum += left.subSum;
        }
        if (right != null)
        {
            subCount += right.subCount;
            subSum += right.subSum;
        }
    }
}

class Treap
{
    TreapNode root;

    static TreapNode RotateRight(TreapNode p)
    {
        var q = p.left;
        p.left = q.right;
        q.right = p;
        p.Recalc();
        q.Recalc();
        return q;
    }

    static TreapNode RotateLeft(TreapNode p)
    {
        var q = p.right;
        p.right = q.left;
        q.left = p;
        p.Recalc();
        q.Recalc();
        return q;
    }

    TreapNode Insert(TreapNode p, int key)
    {
        if (p == null) return new TreapNode(key);

        if (key == p.key)
        {
            p.count++;
            p.Recalc();
            return p;
        }

        if (key < p.key)
        {
            p.left = Insert(p.left, key);
            if (p.left.priority > p.priority)
                p = RotateRight(p);
        }
        else
        {
            p.right = Insert(p.right, key);
            if (p.right.priority > p.priority)
                p = RotateLeft(p);
        }
        p.Recalc();
        return p;
    }

    TreapNode Remove(TreapNode p, int key)
    {
        if (p == null) return null;

        if (key == p.key)
        {
            if (p.count > 1)
            {
                p.count--;
                p.Recalc();
                return p;
            }

            if (p.left == null) return p.right;
            if (p.right == null) return p.left;

            if (p.left.priority > p.right.priority)
            {
                p = RotateRight(p);
                p.right = Remove(p.right, key);
            }
            else
            {
                p = RotateLeft(p);
                p.left = Remove(p.left, key);
            }
        }
        else if (key < p.key)
            p.left = Remove(p.left, key);
        else
            p.right = Remove(p.right, key);

        if (p != null) p.Recalc();
        return p;
    }

    long SumSmallest(TreapNode p, int k)
    {
        if (p == null || k <= 0) return 0;

        int leftCount = p.left?.subCount ?? 0;

        if (k <= leftCount)
            return SumSmallest(p.left, k);

        if (k <= leftCount + p.count)
        {
            long leftSum = p.left?.subSum ?? 0;
            long takeFromHere = (long)(k - leftCount) * p.key;
            return leftSum + takeFromHere;
        }

        long leftSide = p.left?.subSum ?? 0;
        long midSide = (long)p.count * p.key;
        return leftSide + midSide + SumSmallest(p.right, k - leftCount - p.count);
    }

    public void Insert(int x) => root = Insert(root, x);
    public void Remove(int x) => root = Remove(root, x);
    public long SumSmallest(int k) => SumSmallest(root, k);
    public int Count => root?.subCount ?? 0;
}

class Program
{
    static void Main()
    {
        string[] s = Console.ReadLine().Split();
        int n = int.Parse(s[0]);
        int q = int.Parse(s[1]);

        int[] karte = new int[n];
        string[] s1 = Console.ReadLine().Split();
        for (int i = 0; i < n; i++) karte[i] = int.Parse(s1[i]);

        Dictionary<int, int> freq = new Dictionary<int, int>();
        Treap treap = new Treap();

        void AddProduct(int val, int oldFreq, int newFreq)
        {
            if (oldFreq > 0)
                treap.Remove(val * oldFreq);
            if (newFreq > 0)
                treap.Insert(val * newFreq);
        }

        // Initial load
        foreach (int x in karte)
        {
            freq.TryGetValue(x, out int f);
            AddProduct(x, f, f + 1);
            freq[x] = f + 1;
        }

        List<long> answers = new List<long>();

        for (int i = 0; i < q; i++)
        {
            string[] qq = Console.ReadLine().Split();
            int t = int.Parse(qq[0]);

            if (t == 1)
            {
                int k = int.Parse(qq[1]);
                int remove = n - k;
                int keep = treap.Count - remove;
                if (keep <= 0) answers.Add(0);
                else answers.Add(treap.SumSmallest(keep));
            }
            else
            {
                int pos = int.Parse(qq[1]) - 1;
                int newVal = int.Parse(qq[2]);
                int oldVal = karte[pos];

                if (oldVal == newVal) continue;

                // decrease old value
                int fOld = freq[oldVal];
                AddProduct(oldVal, fOld, fOld - 1);
                if (fOld == 1) freq.Remove(oldVal);
                else freq[oldVal] = fOld - 1;

                // increase new value
                freq.TryGetValue(newVal, out int fNew);
                AddProduct(newVal, fNew, fNew + 1);
                freq[newVal] = fNew + 1;

                karte[pos] = newVal;
            }
        }

        foreach (long ans in answers)
            Console.WriteLine(ans);
    }
}